--[[
	BalerBufferFix.lua
	
	Author: 	Azhur
	Date: 		19.11.2022
	Version: 	0.2
	
	Changelog:	v0.1 @29.10.2022 - initial implementation (dirty hack)
                v0.2 @19.11.2022 - minor code changes and cleanup
                v0.3 @30.03.2024 - adeed unfinishedBaleThreshold reduction for JD CP690 and automatic buffer flush before unloading unfinished bale
				---------------------------------------------------------------------------------------------------------------------------
]]

BalerBufferFix = {};
local BalerBufferFix_mt = Class(BalerBufferFix)

function BalerBufferFix.new()
	local self = setmetatable({}, BalerBufferFix_mt)
	self.overloadingTimeLeft = 0
    self.oldUnloadFunc = nil
	return self
end


function BalerBufferFix.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Baler, specializations)
           and SpecializationUtil.hasSpecialization(Foldable, specializations)
end

function BalerBufferFix.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", BalerBufferFix)
    SpecializationUtil.registerEventListener(vehicleType, "onPostLoad", BalerBufferFix)
    SpecializationUtil.registerEventListener(vehicleType, "onPlayAnimation", BalerBufferFix)
    SpecializationUtil.registerEventListener(vehicleType, "onFinishAnimation", BalerBufferFix)

end

function BalerBufferFix:onPostLoad(savegame)
    local spec = self.spec_foldable
    --unregister event listeners for balers without buffer or folding parts
    if #spec.foldingParts == 0 or not self.spec_baler.nonStopBaling then
        SpecializationUtil.removeEventListener(self, "onRegisterActionEvents", BalerBufferFix)
        SpecializationUtil.removeEventListener(self, "onPlayAnimation", BalerBufferFix)
        SpecializationUtil.removeEventListener(self, "onFinishAnimation", BalerBufferFix)
    else
        --register overloads
        Baler.updateActionEvents = Utils.overwrittenFunction(Baler.updateActionEvents, BalerBufferFix.updateActionEvents)
        Baler.actionEventUnloading = Utils.overwrittenFunction(Baler.actionEventUnloading, BalerBufferFix.onActionEventUnloading)
        Baler.onUpdateTick = Utils.appendedFunction(Baler.onUpdateTick, BalerBufferFix.after_onUpdateTick)
    end
    --remove treshhold for minimal bale size for JD CP690
    if self.configFileName == 'data/vehicles/johnDeere/cp690/cp690.xml' then
        self.spec_baler.unfinishedBaleThreshold = 1
    end
end



function BalerBufferFix:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
    local spec = self.spec_baler

    if self.isClient then
        if isActiveForInputIgnoreSelection then
            local _, actionEventId = self:addPoweredActionEvent(spec.actionEvents, InputAction.BALER_FLUSH_BUFFER, self, BalerBufferFix.actionEventFlush, false, true, false, true, nil)
            spec.actionEvents[InputAction.BALER_FLUSH_BUFFER] = actionEventId
            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_HIGH)
            g_inputBinding:setActionEventActive(actionEventId, (spec.unloadingState == Baler.UNLOADING_CLOSED) and self.spec_foldable:getIsUnfolded())
            g_inputBinding:setActionEventTextVisibility(actionEventId, (spec.unloadingState == Baler.UNLOADING_CLOSED) and self.spec_foldable:getIsUnfolded())
        end
    end
end

--disable hotkey at start of folding and bale unload
function BalerBufferFix:onPlayAnimation(animName)
    local spec = self.spec_baler
    local actionEvent = spec.actionEvents[InputAction.BALER_FLUSH_BUFFER]
    if (animName == 'folding')  and (actionEvent ~= nil) and self.spec_foldable:getIsUnfolded() then
       -- unfolded and started to fold, deactivate hotkey
       g_inputBinding:setActionEventActive(actionEvent, false)
       g_inputBinding:setActionEventTextVisibility(actionEvent, false)
--       print ('!folding hide!')
    end

    if (animName == 'unloadBale')  and (actionEvent ~= nil) then
       -- unloading bale, deactivate hotkey
       g_inputBinding:setActionEventActive(actionEvent, false)
       g_inputBinding:setActionEventTextVisibility(actionEvent, false)
       --print ('!unloading hide!')
    end
end

--enable hotkey at end of unfolding and bale unload
function BalerBufferFix:onFinishAnimation(animName)
    local spec = self.spec_baler
    local actionEvent = spec.actionEvents[InputAction.BALER_FLUSH_BUFFER]
    if (animName == 'closeDoor')  and (actionEvent ~= nil) then
       -- unloaded bale, activate hotkey
       g_inputBinding:setActionEventActive(actionEvent, true)
       g_inputBinding:setActionEventTextVisibility(actionEvent, true)
    end

    if (animName == 'folding')  and (actionEvent ~= nil) and self.spec_foldable:getIsUnfolded() then
        -- unfolded, activate hotkey
        g_inputBinding:setActionEventActive(actionEvent, true)
        g_inputBinding:setActionEventTextVisibility(actionEvent, true)
    end
end

function BalerBufferFix:actionEventFlush()
    local spec = self.spec_baler
    if self.spec_turnOnVehicle.isTurnedOn ~= true then
        SpecializationUtil.raiseEvent(self, "onTurnedOn")
        self.rootVehicle:raiseStateChange(Vehicle.STATE_CHANGE_TURN_ON, self)
    end
    --initiate buffer overloading and animation (if any)
    spec.buffer.unloadingStarted = true
    if spec.buffer.overloadAnimation ~= nil then
        self:playAnimation(spec.buffer.overloadAnimation, spec.buffer.overloadAnimationSpeed)
    end
end

--fixes for new bug, introduced with game patch 1.9.0.0, reverting this function to previous version
--Bug: JD CP690 unable to unload unfinished bale (no 'Y' hotkey)
function BalerBufferFix.updateActionEvents(self)
    local spec = self.spec_baler
    local actionEvent = spec.actionEvents[InputAction.IMPLEMENT_EXTRA3]
    if actionEvent ~= nil then
        local showAction = false
        --this line!!!
        if ((spec.hasPlatform and not spec.platformAutomaticDrop) or (not spec.hasPlatform and not spec.automaticDrop)) and self:isUnloadingAllowed() then
            if spec.hasUnloadingAnimation or spec.allowsBaleUnloading then

                if spec.unloadingState == Baler.UNLOADING_CLOSED then
                    if self:getCanUnloadUnfinishedBale() then
                        g_inputBinding:setActionEventText(actionEvent.actionEventId, spec.texts.unloadUnfinishedBale)
                        showAction = true
                    end
                    if #spec.bales > 0 then
                        g_inputBinding:setActionEventText(actionEvent.actionEventId, spec.texts.unloadBaler)
                        showAction = true
                    end
                elseif spec.unloadingState == Baler.UNLOADING_OPEN then
                    if spec.hasUnloadingAnimation then
                        g_inputBinding:setActionEventText(actionEvent.actionEventId, spec.texts.closeBack)
                        showAction = true
                    end
                end
            end
        end

        if spec.platformReadyToDrop then
            g_inputBinding:setActionEventText(actionEvent.actionEventId, spec.texts.unloadBaler)
            showAction = true
        end

        g_inputBinding:setActionEventActive(actionEvent.actionEventId, showAction)
    end

    if spec.toggleableAutomaticDrop then
        actionEvent = spec.actionEvents[InputAction.IMPLEMENT_EXTRA4]
        if actionEvent ~= nil then
            local automaticDropState = spec.automaticDrop
            if spec.hasPlatform then
                automaticDropState = spec.platformAutomaticDrop
            end
            g_inputBinding:setActionEventText(actionEvent.actionEventId, automaticDropState and spec.toggleAutomaticDropTextNeg or spec.toggleAutomaticDropTextPos)
        end
    end

    if #spec.baleTypes > 1 then
        actionEvent = spec.actionEvents[InputAction.TOGGLE_BALE_TYPES]
        if actionEvent ~= nil then
            local baleTypeDef = spec.baleTypes[spec.preSelectedBaleTypeIndex]
            local baleSize
            if spec.hasUnloadingAnimation then
                baleSize = baleTypeDef.diameter
            else
                baleSize = baleTypeDef.length
            end

            g_inputBinding:setActionEventText(actionEvent.actionEventId, spec.changeBaleTypeText:format(baleSize * 100))
        end
    end
end
--end of bugfix

-- this part adds buffer flush at the start of sequence of unloading unfinished bale
function BalerBufferFix:onActionEventUnloading(superFunc, param1, param2, param3, param4)
	local spec = self.spec_baler
    if self:getFillUnitFillLevel(spec.buffer.fillUnitIndex) > 0 then
		--Two-chamber vehicles: trigger unloading from buffer to main chamber berfore unloading unfinished bale, if buffer is not empty
        --print('BUFFER UNLOAD')
        BalerBufferFix.actionEventFlush(self)
        --initializing counter for buffer unloading time
        self.overloadingTimeLeft = spec.buffer.overloadingDuration
        --save pointer to overloaded function, for future use
        self.oldUnloadFunc = superFunc
		-- Ignore the event in this case, don't forward it
	else
		-- Forward the event through base game mechanism in all other cases
		superFunc(self, param1, param2, param3, param4)
	end
end

function BalerBufferFix:after_onUpdateTick(dt, ...)
	-- decrease time counter
    if (self.overloadingTimeLeft ~= nil) and (self.overloadingTimeLeft > 0) then
        self.overloadingTimeLeft = self.overloadingTimeLeft - dt
    end
    --print ('unload time left:', self.overloadingTimeLeft)
    --if counter expired, call original unloading function
    if (self.overloadingTimeLeft ~= nil) and (self.overloadingTimeLeft < 0) then
        self.overloadingTimeLeft = 0
        --print ('unloading time!')
        if self.oldUnloadFunc ~= nil then
            BalerBufferFix.onActionEventUnloading(self, self.oldUnloadFunc)
        end
    end
end
