--[[
Copyright (C) GtX (Andy), 2021

Author: GtX | Andy
Date: 12.04.2021
Revision: FS22-01

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No changes are to be made to this script without permission from GtX | Andy

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
An diesem Skript dürfen ohne Genehmigung von GtX | Andy keine Änderungen vorgenommen werden
]]


SupplyTransportMission = {}
SupplyTransportMission.CATEGORY = 10

local SupplyTransportMission_mt = Class(SupplyTransportMission, AbstractMission)
InitObjectClass(SupplyTransportMission, "SupplyTransportMission")

function SupplyTransportMission.new(isServer, isClient, customMt)
    local self = AbstractMission.new(isServer, isClient, customMt or SupplyTransportMission_mt)

    self.mission = g_currentMission -- Fix for error in AbstractMission 'self.mission' appears to be missing in Version 1.1.0.0!

    self.sellingStation = nil
    self.sellingStationId = nil

    self.pricePerLitre = 0

    self.contractLiters = 0
    self.deliveredLiters = 0

    self.contractDuration = 0
    self.lastFillSoldTime = -1

    self.contractDay = 0
    self.contractTime = 0

    self.npcIndex = 1
    self.fillType = FillType.UNKNOWN

    self.remainingTimeText = g_i18n:getText("ui_pendingMissionTimeLeftTitle") .. ":  %02d:%02d:%02d"

    return self
end

function SupplyTransportMission:delete()
    local sellingStation = self:getSellingStation()

    if sellingStation ~= nil then
        sellingStation.missions[self] = nil
    end

    self:removeMapHotspot()

    SupplyTransportMission:superClass().delete(self)
end

function SupplyTransportMission:saveToXMLFile(xmlFile, key)
    SupplyTransportMission:superClass().saveToXMLFile(self, xmlFile, key)

    local deliveryKey = string.format("%s.delivery", key)

    if self.sellingStation == nil and self.sellingStation.owningPlaceable == nil and self.sellingStation.owningPlaceable.currentSavegameId == nil then
        Logging.xmlWarning(xmlFile, "Failed to save Supply & Transport Contract at '%s', sellingStation could not be found!", deliveryKey)

        return
    end

    local owningPlaceable = self.sellingStation.owningPlaceable
    local unloadingStationIndex = g_currentMission.storageSystem:getPlaceableUnloadingStationIndex(owningPlaceable, self.sellingStation) -- ??

    if unloadingStationIndex == nil then
        Logging.xmlWarning(xmlFile, "Failed to save Supply & Transport Contract at '%s', unloading station index for sellingStation '%s' could not be found!", deliveryKey, self:getSellingStationTitle())

        return
    end

    local fillTypeName = g_fillTypeManager:getFillTypeNameByIndex(self.fillType)

    if fillTypeName == nil then
        Logging.xmlWarning(xmlFile, "Failed to save Supply & Transport Contract at '%s', fillType could not be found!", deliveryKey)

        return
    end

    -- Keep key names similar to other missions types
    setXMLInt(xmlFile, deliveryKey .. "#sellPointPlaceableId", owningPlaceable.currentSavegameId)
    setXMLInt(xmlFile, deliveryKey .. "#unloadingStationIndex", unloadingStationIndex)

    setXMLFloat(xmlFile, deliveryKey .. "#pricePerLitre", self.pricePerLitre)
    setXMLFloat(xmlFile, deliveryKey .. "#expectedLiters", self.contractLiters)

    if self.deliveredLiters > 0 then
        setXMLFloat(xmlFile, deliveryKey .. "#depositedLiters", self.deliveredLiters)
    end

    setXMLInt(xmlFile, deliveryKey .. "#contractDay", self.contractDay)
    setXMLFloat(xmlFile, deliveryKey .. "#contractTime", MathUtil.msToMinutes(self.contractTime))

    setXMLFloat(xmlFile, deliveryKey .. "#contractDuration", self.contractDuration)

    setXMLInt(xmlFile, deliveryKey .. "#npcIndex", self.npcIndex)
    setXMLString(xmlFile, deliveryKey .. "#fillType", fillTypeName)
end

function SupplyTransportMission:loadFromXMLFile(xmlFile, key)
    if not SupplyTransportMission:superClass().loadFromXMLFile(self, xmlFile, key) then
        return false
    end

    local sellPointPlaceableId = getXMLInt(xmlFile, key .. ".delivery#sellPointPlaceableId")
    local unloadingStationIndex = getXMLInt(xmlFile, key .. ".delivery#unloadingStationIndex")

    if sellPointPlaceableId == nil or unloadingStationIndex == nil then
        Logging.xmlError(xmlFile, "Supply & Transport Contract sellPoint at '%s.delivery' is no longer available, mission will be ignored!", key)

        return false
    end

    local sellPointPlaceable = g_currentMission.placeableSystem:getPlaceableBySavegameId(sellPointPlaceableId)

    if sellPointPlaceable == nil then
        Logging.xmlError(xmlFile, "Supply & Transport Contract sellPoint at '%s.delivery' is no longer available, mission will be ignored!", key)

        return false
    end

    local sellingStation = g_currentMission.storageSystem:getPlaceableUnloadingStation(sellPointPlaceable, unloadingStationIndex)

    if sellingStation == nil then
        Logging.xmlError(xmlFile, "Supply & Transport Contract sellPoint at '%s.delivery' is no longer available, mission will be ignored!", key)

        return false
    end

    local fillTypeStr = getXMLString(xmlFile, key .. ".delivery#fillType")
    local fillType = g_fillTypeManager:getFillTypeIndexByName(fillTypeStr)

    if fillType == nil then
        local savegameDirectory = g_currentMission.missionInfo.savegameDirectory or ""

        Logging.xmlError(xmlFile, "Supply & Transport Contract fill type '%s' is no longer available, mission will be ignored!", tostring(fillTypeStr))

        return false
    end

    self.sellingStation = sellingStation
    self.sellingStation.missions[self] = self

    self.fillType = fillType
    self.pricePerLitre = getXMLFloat(xmlFile, key .. ".delivery#pricePerLitre")

    if self.pricePerLitre == nil or self.pricePerLitre <= 0 then
        self.pricePerLitre = self.sellingStation:getEffectiveFillTypePrice(fillType)

        if self.pricePerLitre == nil or self.pricePerLitre <= 0 then
            return false
        end
    end

    self.deliveredLiters = getXMLFloat(xmlFile, key .. ".delivery#depositedLiters") or 0
    self.contractLiters = getXMLFloat(xmlFile, key .. ".delivery#expectedLiters") or self.deliveredLiters

    if self.contractLiters <= 0 then
        return false
    end

    self.contractDay = getXMLInt(xmlFile, key .. ".delivery#contractDay") or 1
    self.contractTime = MathUtil.minutesToMs(Utils.getNoNil(getXMLFloat(xmlFile, key .. ".delivery#contractTime"), 0))
    self.contractDuration = getXMLFloat(xmlFile, key .. ".delivery#contractDuration") or 60

    self.npcIndex = getXMLInt(xmlFile, key .. ".delivery#npcIndex")

    if self:getNPC() == nil then
        local deliveryMissionManager = g_missionManager.deliveryMissionManager

        if deliveryMissionManager ~= nil and deliveryMissionManager.getRandomNpcIndex ~= nil then
            self.npcIndex = deliveryMissionManager:getRandomNpcIndex()
        else
            self.npcIndex = g_npcManager:getRandomIndex()
        end
    end

    if self.status == AbstractMission.STATUS_FINISHED and not self.success then
        self.reimbursement = self.pricePerLitre * self.deliveredLiters
    end

    return true
end

function SupplyTransportMission:writeStream(streamId, connection)
    SupplyTransportMission:superClass().writeStream(self, streamId, connection)

    NetworkUtil.writeNodeObjectId(streamId, NetworkUtil.getObjectId(self.sellingStation))

    streamWriteUIntN(streamId, self.fillType, FillTypeManager.SEND_NUM_BITS)
    streamWriteFloat32(streamId, self.contractLiters)

    streamWriteInt32(streamId, self.contractDay)
    streamWriteFloat32(streamId, self.contractTime)

    streamWriteFloat32(streamId, self.contractDuration)
    streamWriteUInt8(streamId, self.npcIndex)

    if self.status == AbstractMission.STATUS_FINISHED and not self.success then
        streamWriteFloat32(streamId, self.reimbursement or 0)
    end
end

function SupplyTransportMission:readStream(streamId, connection)
    SupplyTransportMission:superClass().readStream(self, streamId, connection)

    self.sellingStationId = NetworkUtil.readNodeObjectId(streamId)

    self.fillType = streamReadUIntN(streamId, FillTypeManager.SEND_NUM_BITS)
    self.contractLiters = streamReadFloat32(streamId)

    self.contractDay = streamReadInt32(streamId)
    self.contractTime = streamReadFloat32(streamId)

    self.contractDuration = streamReadFloat32(streamId)
    self.npcIndex = streamReadUInt8(streamId)

    if self.status == AbstractMission.STATUS_FINISHED and not self.success then
        self.reimbursement = streamReadFloat32(streamId)
    end
end

function SupplyTransportMission:update(dt)
    SupplyTransportMission:superClass().update(self, dt)

    if self.mapHotspot == nil and g_currentMission.player.farmId == self.farmId and self.status == AbstractMission.STATUS_RUNNING then
        local sellingStation = self:getSellingStation()

        if sellingStation ~= nil then
            local mapHotspot = MissionHotspot.new()

            if sellingStation.mapHotspot == nil then
                local x, _, z = getWorldTranslation(self.sellingStation.owningPlaceable.rootNode)
                mapHotspot:setWorldPosition(x, z)
            else
                mapHotspot:setWorldPosition(sellingStation.mapHotspot:getWorldPosition())
            end

            g_currentMission:addMapHotspot(mapHotspot)
            self.mapHotspot = mapHotspot
        end
    end

    if self.lastFillSoldTime > 0 then
        self.lastFillSoldTime = self.lastFillSoldTime - 1

        if self.lastFillSoldTime == 0 then
            local missionProgressText = g_i18n:getText("supplyTransport_missionProgress", self.customEnvironment)
            local percent = math.floor((self.completion * 100) + 0.5)

            g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_OK, string.format(missionProgressText, percent, self:getFillTypeTitle()))
        end
    end
end

function SupplyTransportMission:init(fillType, contractLiters, npcIndex)
    if not SupplyTransportMission:superClass().init(self) or (fillType == nil) then
        return false
    end

    local sellingStation, pricePerLitre = nil, 0

    for _, unloadingStation in pairs(g_currentMission.storageSystem:getUnloadingStations()) do
        if unloadingStation.isSellingPoint and unloadingStation.allowMissions and unloadingStation.owningPlaceable ~= nil and unloadingStation.acceptedFillTypes[fillType] then
            local price = unloadingStation:getEffectiveFillTypePrice(fillType)

            if pricePerLitre < price then
                pricePerLitre = price
                sellingStation = unloadingStation
            end
        end
    end

    if sellingStation == nil or pricePerLitre <= 0 then
        return false
    end

    self.fillType = fillType
    self.deliveredLiters = 0

    self.sellingStation = sellingStation
    self.pricePerLitre = pricePerLitre

    self.contractDay = g_currentMission.environment.currentDay
    self.contractTime = g_currentMission.environment.dayTime

    self.contractLiters = contractLiters or (math.random(5, 250) * 1000)
    self.contractDuration = math.random(1, 48) * 60

    self.npcIndex = g_npcManager:getNPCByIndex(npcIndex) ~= nil and npcIndex or 1
    self.reward = self.contractLiters * (pricePerLitre * self:getRewardMultiplier())

    self.missingProductWarning = g_i18n:getText("supplyTransport_missingProductWarning", self.customEnvironment)

    return true
end

function SupplyTransportMission:start()
    local sellingStation = self:getSellingStation()

    if (sellingStation == nil) or not SupplyTransportMission:superClass().start(self) then
        return false
    end

    sellingStation.missions[self] = self -- Add mission to Selling Station

    return true
end

function SupplyTransportMission:finish(success)
    self.success = Utils.getNoNil(success, false)
    self.status = AbstractMission.STATUS_FINISHED

    if self.isServer then
        local reimbursement = 0
        local stealingCost = 0

        if not self.success then
            reimbursement = self.pricePerLitre * self.deliveredLiters

            if not self.sellpointDeleted then
                stealingCost = self.pricePerLitre * (self.contractLiters - self.deliveredLiters)
            end
        end

        self.reimbursement = reimbursement
        self.stealingCost = stealingCost

        g_server:broadcastEvent(SupplyTransportMissionFinishedEvent.new(self, self.success, self.reimbursement, self.stealingCost))
    end

    g_messageCenter:publish(MissionFinishedEvent, {self, self.success, self.reimbursement, self.stealingCost}) -- Notify listeners, this also updates GUI if open

    self:removeMapHotspot() -- Delete the map hotspot

    local sellingStation = self:getSellingStation()

    if sellingStation ~= nil then
        sellingStation.missions[self] = nil -- Remove mission from the Selling Station
    end

    if self.success then
        g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_OK, g_i18n:getText("supplyTransport_missionCompleted", self.customEnvironment)) -- Success
    else
        if self.sellpointDeleted then
            g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_CRITICAL, g_i18n:getText("supplyTransport_missionCancelled", self.customEnvironment)) -- Placeable Sellpoint was removed
        else
            g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_CRITICAL, g_i18n:getText("supplyTransport_missionFailed", self.customEnvironment)) -- Time expired or user cancelled
        end
    end
end

function SupplyTransportMission:dismiss()
    if self.isServer then
        local moneyChange = self.reward

        if not self.success then
            local stealingCost = self.stealingCost or 0

            if self.reimbursement ~= nil and self.reimbursement > 0 then
                moneyChange = self.reimbursement - stealingCost
            else
                moneyChange = -stealingCost
            end
        end

        if moneyChange ~= nil and moneyChange ~= 0 then
            g_currentMission:addMoney(moneyChange, self.farmId, MoneyType.MISSIONS, true, true)
        end
    end
end

function SupplyTransportMission:fillSold(delta)
    self.deliveredLiters = math.min(self.deliveredLiters + delta, self.contractLiters)

    if self.deliveredLiters >= self.contractLiters and self.sellingStation ~= nil then
        self.sellingStation.missions[self] = nil -- Remove mission from the Selling Station and start selling
    end

    self.lastFillSoldTime = 30 -- Reset notification timer
end

function SupplyTransportMission:getRewardMultiplier()
    if g_currentMission.missionInfo.economicDifficulty == 1 then
        return 1.6 -- New Farmer
    elseif g_currentMission.missionInfo.economicDifficulty == 2 then
        return 1.4 -- Farm Manager
    end

    return 1.2 -- Start From Scratch
end

function SupplyTransportMission:getRemainingTime(formatToString)
    local environment = g_currentMission.environment

    local usedDaysToMinutes = (environment.currentDay - self.contractDay) * 24 * 60
    local usedMinutes = MathUtil.msToMinutes(environment.dayTime - self.contractTime)

    local remainingMinutes = self.contractDuration - (usedDaysToMinutes + usedMinutes)

    if formatToString then
        if remainingMinutes > 0 then
            local hours = math.floor(remainingMinutes / 60)
            local minutes = remainingMinutes - hours * 60
            local seconds = (remainingMinutes - math.floor(remainingMinutes)) * 60

            return string.format(self.remainingTimeText, hours, minutes, seconds)
        end

        return string.format(self.remainingTimeText, 0, 0, 0)
    end

    return remainingMinutes
end

function SupplyTransportMission:getNPC()
    return g_npcManager:getNPCByIndex(self.npcIndex)
end

function SupplyTransportMission:getReward()
    return self.reward
end

function SupplyTransportMission:getData()
    local fillTypeTitle = self:getFillTypeTitle()
    local litres = string.format("%s %s", g_i18n:formatNumber(g_i18n:getVolume(self.contractLiters), 0), g_i18n:getVolumeUnit(true))

    return {
        location = fillTypeTitle,
        jobType = g_i18n:getText("supplyTransport_dataTitle", self.customEnvironment),
        action = "",
        timeLimit = tostring(self:getRemainingTime(true)),
        description = string.format(g_i18n:getText("supplyTransport_dataDescription", self.customEnvironment), litres, fillTypeTitle, self:getSellingStationTitle())
    }
end

function SupplyTransportMission:getCompletion()
    return self.deliveredLiters / self.contractLiters
end

function SupplyTransportMission:getFillTypeTitle()
    local fillTypeDesc = g_fillTypeManager:getFillTypeByIndex(self.fillType)

    return fillTypeDesc ~= nil and fillTypeDesc.title or ""
end

function SupplyTransportMission:getSellingStationTitle()
    local sellingStation = self:getSellingStation()

    return sellingStation ~= nil and sellingStation:getName() or "Unknown"
end

function SupplyTransportMission:getSellingStation()
    -- Allow the client to resolve late to avoid missing or incomplete sellingStation
    if self.sellingStationId ~= nil then
        self.sellingStation = NetworkUtil.getObject(self.sellingStationId)

        if self.sellingStation ~= nil then
            self.sellingStationId = nil
        end
    end

    return self.sellingStation
end

function SupplyTransportMission:removeMapHotspot()
    if self.mapHotspot ~= nil then
        g_currentMission:removeMapHotspot(self.mapHotspot)

        self.mapHotspot:delete()
        self.mapHotspot = nil
    end
end

g_missionManager:registerMissionType(SupplyTransportMission, "supplyTransport", SupplyTransportMission.CATEGORY, 1)
