--[[
Copyright (C) GtX (Andy), 2021

Author: GtX | Andy
Date: 12.04.2021
Revision: FS22-01

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No changes are to be made to this script without permission from GtX | Andy

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
An diesem Skript dürfen ohne Genehmigung von GtX | Andy keine Änderungen vorgenommen werden
]]


SupplyTransportMissionManager = {}
SupplyTransportMissionManager.MAX_MISSIONS = 4 -- This can be increased after saving game for first time in the 'missions.xml' at 'missions.maxSupplyTransportContracts' or with the console command.

local SupplyTransportMissionManager_mt = Class(SupplyTransportMissionManager)
local isDevelopmentVersion = false

function SupplyTransportMissionManager.new(isServer, isClient, buildId, versionString, loadConsoleCommands, modName, modDirectory, missionManager)
    local self = setmetatable({}, SupplyTransportMissionManager_mt)

    self.isServer = isServer
    self.isClient = isClient

    self.buildId = buildId
    self.versionString = versionString

    self.modName = modName
    self.modDirectory = modDirectory

    self.missionManager = missionManager

    self.missions = {}
    self.invalidFillTypes = {}

    g_messageCenter:subscribe(MessageType.UNLOADING_STATIONS_CHANGED, self.onUnloadingStationsChanged, self)

    -- Testing and content creator commands for SP or Hosted MP only. Use -cheats or -consoleCommandsGtX start parameter
    if self.isServer and (loadConsoleCommands or g_addCheatCommands) then
        addConsoleCommand("gtxSupplyTransportGenerateMissions", "Generate missions for testing only [clearExistingContracts]", "consoleCommandGenerateMissions", self)
        addConsoleCommand("gtxSupplyTransportSetMaxMissions", "Set maximum number of missions that will be generated [maxMissions (Leave blank to reset)]", "consoleCommandSetMaxMissions", self)
    end

    return self
end

function SupplyTransportMissionManager:loadMapData(missionManager, xmlFile)
    if self.isServer then
        self.invalidFillTypes = {
            [FillType.TARP] = true,
            [FillType.AIR] = true,
            [FillType.ELECTRICCHARGE] = true,
            [FillType.ROUNDBALE] = true,
            [FillType.SQUAREBALE] = true
        }

        for _, fillType in pairs(g_fillTypeManager:getFillTypes()) do
            if not self.invalidFillTypes[fillType.index] then
                local name = fillType.name

                if name:sub(1, 10) == "ROUNDBALE_" or name:sub(1, 11) == "SQUAREBALE_" then
                    self.invalidFillTypes[fillType.index] = true
                end
            end
        end

        if isDevelopmentVersion then
            for fillTypeIndex, _ in pairs(self.invalidFillTypes) do
                local fillType = g_fillTypeManager:getFillTypeByIndex(fillTypeIndex)

                print(string.format("  DevInfo: [SupplyTransportMissionManager] Ignoring invalid fillType (%d) | (%s) | (%s)", fillTypeIndex, fillType.name, fillType.title))
            end
        end
    end
end

function SupplyTransportMissionManager:unloadMapData(missionManager)
    g_messageCenter:unsubscribeAll(self)

    if self.isServer then
        removeConsoleCommand("gtxSupplyTransportGenerateMissions")
        removeConsoleCommand("gtxSupplyTransportSetMaxMissions")
    end
end

function SupplyTransportMissionManager:generateMissions(dt)
    local maxMissions = self:getMaxMissions()

    if #self.missions < maxMissions then
        local missionType = self.missionManager:getMissionType("supplyTransport")
        local missionsGenerated = false

        if missionType ~= nil then
            local generationAttempts = 0

            repeat
                local fillTypeIndex = self:getRandomFilltype()

                generationAttempts = generationAttempts + 1

                if fillTypeIndex ~= nil then
                    local mission = missionType.classObject.new(self.isServer, self.isClient)

                    mission.type = missionType
                    mission.customEnvironment = self.modName -- Localise texts

                    if mission:init(fillTypeIndex, self:getRandomLitres(fillTypeIndex), self:getRandomNpcIndex()) then
                        self.missionManager:assignGenerationTime(mission)
                        mission:register()

                        table.insert(self.missionManager.missions, mission)
                        table.insert(self.missions, mission)

                        missionsGenerated = true
                        generationAttempts = 0

                        if isDevelopmentVersion then
                            local fillTypeTitle = mission:getFillTypeTitle()
                            local sellingStationTitle = mission:getSellingStationTitle()
                            local sellPrice = g_i18n:formatMoney(mission.pricePerLitre * 1000, 0, true, true)
                            local reward = g_i18n:formatMoney(mission.reward, 0, true, true)
                            local profit = g_i18n:formatMoney(mission.reward - (mission.contractLiters * mission.pricePerLitre), 0, true, true)
                            local text = "Delivery mission created  -  Sell Point: %s | Fill Type: %s | Sell Price: %s | Reward: %s | Profit: %s"

                            print("  DevInfo: [SupplyTransportMissionManager] " .. string.format(text, sellingStationTitle, fillTypeTitle, sellPrice, reward, profit))
                        end
                    else
                        mission:delete()
                    end
                end
            until (generationAttempts >= 6) or (#self.missions >= maxMissions)
        end

        if missionsGenerated then
            g_messageCenter:publish(MessageType.MISSION_GENERATED, {self})
        end
    end
end

function SupplyTransportMissionManager:updateMissions(dt)
    for _, mission in ipairs(self.missions) do
        if mission.getRemainingTime ~= nil and mission:getRemainingTime() <= 0 then
            if mission.status == AbstractMission.STATUS_RUNNING then
                mission:finish(false) -- Fail the mission
            elseif mission.status == AbstractMission.STATUS_STOPPED then
                mission:delete() -- Never accepted

                break
            end
        end
    end
end

function SupplyTransportMissionManager:addMissionToList(mission)
    table.addElement(self.missions, mission)
end

function SupplyTransportMissionManager:removeMissionFromList(mission)
    table.removeElement(self.missions, mission)
end

function SupplyTransportMissionManager:onUnloadingStationsChanged()
    local unloadingStations = g_currentMission.storageSystem:getUnloadingStations()

    for i = #self.missions, 1, -1 do
        local mission = self.missions[i]

        if mission.sellingStation ~= nil and unloadingStations[mission.sellingStation] == nil then
            mission.sellpointDeleted = true -- Make sure no penalty is applied

            if mission.status == AbstractMission.STATUS_RUNNING then
                if isDevelopmentVersion then
                    print(string.format("  DevInfo: [SupplyTransportMissionManager] Selling Point '%s' for 'Delivery Mission' has been deleted, contract cancelled!", mission.sellingStation:getName()))
                end

                mission:finish(false) -- Fail the mission but no penalty for missing crops, just payout crops delivered
            else
                if isDevelopmentVersion then
                    print(string.format("  DevInfo: [SupplyTransportMissionManager] Selling Point '%s' for 'Delivery Mission' has been deleted, contract removed from list!", mission.sellingStation:getName()))
                end

                mission:delete() -- Delete the mission so it is removed from list
            end
        end
    end
end

function SupplyTransportMissionManager:getRandomFilltype()
    local cropFillTypeMissions = 0
    local cropFillTypesOnly = true

    local numFillTypes = 0
    local validFillTypes = {}
    local ignoreFillTypes = {}

    for _, mission in ipairs (self.missions) do
        local fillTypeIndex = mission.fillType

        if g_fruitTypeManager:getFruitTypeIndexByFillTypeIndex(fillTypeIndex) ~= nil then
            cropFillTypeMissions = cropFillTypeMissions + 1
        end

        -- Only one of each fillType
        ignoreFillTypes[fillTypeIndex] = true
    end

    -- Make sure half of the missions are crop fillTypes
    if cropFillTypeMissions >= self:getMaxMissions() * 0.5 then
        cropFillTypesOnly = false
    end

    for _, unloadingStation in pairs(g_currentMission.storageSystem:getUnloadingStations()) do
        if unloadingStation.isSellingPoint and unloadingStation.allowMissions and unloadingStation.owningPlaceable ~= nil then
            for fillTypeIndex, _ in pairs (unloadingStation.acceptedFillTypes) do
                if not ignoreFillTypes[fillTypeIndex] and not self.invalidFillTypes[fillTypeIndex] then
                    ignoreFillTypes[fillTypeIndex] = true

                    if not cropFillTypesOnly or g_fruitTypeManager:getFruitTypeIndexByFillTypeIndex(fillTypeIndex) ~= nil then
                        numFillTypes = numFillTypes + 1
                        validFillTypes[numFillTypes] = fillTypeIndex
                    end
                end
            end
        end
    end

    return validFillTypes[math.random(numFillTypes)]
end

function SupplyTransportMissionManager:getRandomLitres(fillTypeIndex)
    local halfMaxMissions = self:getMaxMissions() * 0.5
    local numSmallMission = 0

    if fillTypeIndex == FillType.COTTON then
        return math.random(1, 8) * 20000 -- 20000 > 160000
    end

    -- If not cotton then make sure there is a mission that is 50000 litres or less
    for _, mission in ipairs (self.missions) do
        if mission.contractLiters <= 50000 then
            numSmallMission = numSmallMission + 1

            if numSmallMission >= halfMaxMissions then
                return math.random(5, 250) * 1000 -- 5000 > 250000
            end
        end
    end

    return math.random(5, 50) * 1000 -- 5000 > 50000
end

function SupplyTransportMissionManager:getRandomNpcIndex()
    local validNPCS = {}
    local numNpcs = g_npcManager.numNpcs

    for index = 1, numNpcs do
        local valid = true

        for _, mission in ipairs (self.missions) do
            if mission.npcIndex == index then
                valid = false
                break
            end
        end

        if valid then
            table.insert(validNPCS, index)
        end
    end

    return validNPCS[math.random(#validNPCS)] or math.random(numNpcs)
end

function SupplyTransportMissionManager:getMaxMissions()
    return math.min(SupplyTransportMissionManager.MAX_MISSIONS or 4, 8)
end

function SupplyTransportMissionManager:setMaxMissions(maxMissions)
    SupplyTransportMissionManager.MAX_MISSIONS = MathUtil.clamp(maxMissions or 4, 1, 8)
end

function SupplyTransportMissionManager:setIsDevelopmentVersion(state)
    isDevelopmentVersion = Utils.getNoNil(state, false)

    if isDevelopmentVersion then
        Logging.info("Development mode active for mod %s - Version %s (b%d)", self.modName, self.versionString, self.buildId)
    end
end

function SupplyTransportMissionManager:consoleCommandSetMaxMissions(maxMissions)
    if self.isServer then
        self:setMaxMissions(tonumber(maxMissions))

        return string.format("Max Supply & Transport Contracts set to %d.", self:getMaxMissions())
    end

    return "'stSetMaxMissions' is a server only command"
end

function SupplyTransportMissionManager:consoleCommandGenerateMissions(clearExistingContracts)
    if self.isServer then
        local numRemoved = 0
        local numAvailable = 0
        local activeMissions = 0

        local clearMissions = Utils.stringToBoolean(clearExistingContracts)

        for i = #self.missions, 1, -1 do
            local mission = self.missions[i]

            if mission ~= nil then
                if mission.status == AbstractMission.STATUS_STOPPED then
                    if clearMissions then
                        mission:delete()
                        numRemoved = numRemoved + 1
                    else
                        numAvailable = numAvailable + 1
                    end
                else
                    activeMissions = activeMissions + 1
                end
            end
        end

        self:generateMissions(g_currentDt)

        return string.format("%d contracts have been cleared, %d new contracts created.", numRemoved, (#self.missions - numAvailable) - activeMissions)
    else

    end

    return "'stGenerateMissions' is a server only command"
end
